// Copyright 2025 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0
import { Button, Card, CardContent, CardHeader, CardTitle } from '@bloodhoundenterprise/doodleui';
import '@neo4j-cypher/codemirror/css/cypher-codemirror.css';
import { CypherEditor } from '@neo4j-cypher/react-codemirror';
import { SeedTypeCypher } from 'js-client-library';
import { Dispatch, FC, SetStateAction, useCallback, useContext, useEffect, useRef, useState } from 'react';
import { useQuery } from 'react-query';
import { useLocation } from 'react-router-dom';
import { graphSchema } from '../../../constants';
import { usePZPathParams } from '../../../hooks';
import { apiClient, cn } from '../../../utils';
import { adaptClickHandlerToKeyDown } from '../../../utils/adaptClickHandlerToKeyDown';
import RuleFormContext from '../Save/RuleForm/RuleFormContext';

const emptyFunction = () => {};

export const PrivilegeZonesCypherEditor: FC<{
    preview?: boolean;
    initialInput?: string;
    onChange?: (content: string) => void;
    stalePreview?: boolean;
    setStalePreview?: Dispatch<SetStateAction<boolean>>;
}> = ({ preview = true, initialInput = '', onChange, stalePreview = false, setStalePreview = () => {} }) => {
    const [cypherQuery, setCypherQuery] = useState(initialInput);
    const [showLabelWarning, setShowLabelWarning] = useState(initialInput?.includes(':Tag_'));

    const cypherEditorRef = useRef<CypherEditor | null>(null);

    const dispatch = useContext(RuleFormContext).dispatch || emptyFunction;
    const { hasZoneId } = usePZPathParams();

    const location = useLocation();
    const receivedQuery = location.state?.query;

    useEffect(() => {
        if (!preview && typeof receivedQuery === 'string' && receivedQuery.length) {
            setCypherQuery(receivedQuery);
        }
    }, [preview, receivedQuery]);

    const kindsQuery = useQuery({
        queryKey: ['graph-kinds'],
        queryFn: ({ signal }) => apiClient.getKinds({ signal }).then((res) => res.data.data.kinds),
    });

    const schema = useCallback(() => graphSchema(kindsQuery.data), [kindsQuery.data]);

    const handleCypherSearch = useCallback(() => {
        if (preview) return;

        if (cypherQuery) {
            setStalePreview(false);
            dispatch({ type: 'set-seeds', seeds: [{ type: SeedTypeCypher, value: cypherQuery }] });
        }
    }, [cypherQuery, preview, dispatch, setStalePreview]);

    const onValueChanged = useCallback(
        (value: string) => {
            if (preview) return;
            setCypherQuery(value);
            setStalePreview(true);

            if (hasZoneId && value.includes(':Tag_')) setShowLabelWarning(true);
            else setShowLabelWarning(false);
        },
        [preview, setCypherQuery, hasZoneId, setStalePreview]
    );

    const setFocusOnCypherEditor = () => cypherEditorRef.current?.cypherEditor.focus();

    useEffect(() => {
        if (typeof onChange === 'function') {
            onChange(cypherQuery);
        }
    }, [cypherQuery, onChange]);

    return (
        <Card>
            <CardHeader>
                <div className='flex justify-between items-center px-6 pt-3'>
                    <CardTitle>{preview ? 'Cypher Preview' : 'Cypher Rule'}</CardTitle>
                </div>
                {!preview && (
                    <p className='px-6 pt-3'>
                        Run your Cypher query first to check results. Once it's run successfully, you’ll be able to save
                        this selector.
                    </p>
                )}
                <p className='italic text-sm px-5 py-2'>
                    Note: The sample results from running this cypher search may include additional entities that are
                    not directly associated with the cypher query due to default rule expansion. In contrast, 'View in
                    Explore' will show only the entities that are directly associated with the cypher query.
                </p>
            </CardHeader>
            <CardContent className='px-6' data-testid='privilege-zones_cypher-container'>
                <div
                    onClick={setFocusOnCypherEditor}
                    onKeyDown={adaptClickHandlerToKeyDown(setFocusOnCypherEditor)}
                    tabIndex={0}
                    className='flex-1'
                    role='textbox'>
                    <CypherEditor
                        className={cn(
                            '[&_.cm-content]:saturate-200 flex flex-col border-solid border border-neutral-5 bg-white dark:bg-[#002b36] rounded-lg min-h-64 overflow-auto grow-1',
                            {
                                'bg-transparent [&_.cm-editor]:bg-transparent [&_.cm-editor_.cm-gutters]:bg-transparent [&_.cm-editor_.cm-gutters]:border-transparent dark:bg-transparent dark:[&_.cm-editor]:bg-transparent dark:[&_.cm-editor_.cm-gutters]:bg-transparent dark:[&_.cm-editor_.cm-gutters]:border-transparent':
                                    preview,
                                'md:min-h-[16rem] md:max-h-[16rem] h-[24rem] max-h-[24rem]': !preview,
                            }
                        )}
                        ref={cypherEditorRef}
                        value={preview ? initialInput : cypherQuery}
                        onValueChanged={onValueChanged}
                        theme={document.documentElement.classList.contains('dark') ? 'dark' : 'light'}
                        schema={schema()}
                        readOnly={preview}
                        autofocus={false}
                        placeholder='Cypher Query'
                        tooltipAbsolute={false}
                        lineWrapping
                        lint
                    />
                </div>
                {showLabelWarning && (
                    <p className='text-error text-sm p-2'>
                        Privilege Zone labels should only be used in cypher within the Explore page. Utilizing Privilege
                        Zone labels in a cypher based Rule seed may result in incomplete data.
                    </p>
                )}
                <div className='flex justify-end items-center'>
                    {!preview && (
                        <Button
                            disabled={!cypherQuery}
                            data-testid='privilege-zones_save_selector-form_update-results-button'
                            className={cn('mt-3', {
                                'animate-pulse': stalePreview,
                            })}
                            onClick={handleCypherSearch}>
                            Run
                        </Button>
                    )}
                </div>
            </CardContent>
        </Card>
    );
};
